import subprocess
import re
import numpy as np
import os
import sys

# --- 設定 ---

# f2pyビルドに使用したMSYS2/MinGW-w64 Pythonへの絶対パス
MSYS_PYTHON = r"D:\msys64\mingw64\bin\python.exe"

# --- 検証設定 ---
# numpy.allcloseで使用する許容誤差
# |absolute(a - b)| <= (atol + rtol * absolute(b))
RTOL = 1e-8  # 相対許容誤差
ATOL = 1e-8  # 絶対許容誤差

# 検証の基準とするベンチマーク
BASE_TARGET_NAME = "Fortran (Original)"

# 検証対象のベンチマークリスト
BENCHMARK_TARGETS = {
    "Fortran (Original)": f".{os.path.sep}bin{os.path.sep}benchmark_svd_fortran_original.exe",
    "Fortran (Refactored)": f".{os.path.sep}bin{os.path.sep}benchmark_svd_fortran_refactored.exe",
    "Python (base)": f"{sys.executable} ./benchmarks/benchmark_svd_python_base.py",
    "Python (Vectorized)": f"{sys.executable} ./benchmarks/benchmark_svd_python_vectorized.py",
    "Python (Base+JIT)": f"{sys.executable} ./benchmarks/benchmark_svd_python_base_jit.py",
    "Python (Base+JIT-Cached)": f"{sys.executable} ./benchmarks/benchmark_svd_python_base_jit_cached.py",
#    "Python (f2py)": f"{MSYS_PYTHON} ./benchmarks/benchmark_svd_python_f2py.py",
    "Python (f2py)": f"{sys.executable} ./benchmarks/benchmark_svd_python_f2py.py",
    "Python (f2py_intent_out)": f"{sys.executable} ./benchmarks/benchmark_svd_python_f2py_intent_out.py",
    "Python (ctypes)": f"{sys.executable} ./benchmarks/benchmark_svd_python_ctypes.py",
}

# デフォルトの検証データファイル
DEFAULT_TEST_DATA_FILE = f".{os.path.sep}test_data{os.path.sep}performance_input{os.path.sep}matrix_100x100_rand_uniform.bin"

def run_and_parse_output(command, test_data_file):
    """
    指定されたコマンドとデータファイルを実行し、stderrから特異値と固有ベクトルを抽出する。
    """
    try:
        args = command.split()
        args.append(test_data_file)
        result = subprocess.run(args, capture_output=True, text=True, check=True, encoding='utf-8')
        stderr = result.stderr

        # --- 特異値の抽出 ---
        s_pattern = re.compile(r"Top 5 singular values.*?\n(.*?)\n", re.DOTALL)
        s_match = s_pattern.search(stderr)
        s_values = np.array([float(val) for val in s_match.group(1).split()]) if s_match else None

        # --- 固有ベクトルの抽出 ---
        u_pattern = re.compile(r"First 3 components of leading eigenvector.*?\n(.*?)\n", re.DOTALL)
        u_match = u_pattern.search(stderr)
        u_vector = np.array([float(val) for val in u_match.group(1).split()]) if u_match else None

        if s_values is None or u_vector is None:
            print(f"[Warning] Could not parse singular values or eigenvector from stderr for command: '{' '.join(args)}'")
            print(f"stderr:\n{stderr}")
            return None
        
        return {"s_values": s_values, "u_vector": u_vector}

    except FileNotFoundError:
        print(f"[ERROR] Command not found: {command.split()[0]}")
        return None
    except subprocess.CalledProcessError as e:
        print(f"[ERROR] Command failed: {' '.join(args)}")
        print(f"stderr:\n{e.stderr}")
        return None
    except Exception as e:
        print(f"[ERROR] An unexpected error occurred: {e}")
        return None

def calculate_relative_error(base, target):
    """二つのNumpy配列間の最大相対誤差を計算する"""
    abs_diff = np.abs(base - target)
    rel_err = np.divide(abs_diff, np.abs(base), out=np.zeros_like(abs_diff, dtype=float), where=base!=0)
    return np.max(rel_err)

def main():
    """
    全てのベンチマークを実行し、特異値と主要な固有ベクトルの一致を検証する。
    """
    # ... (ファイルの存在チェックなどは変更なし) ...
    if len(sys.argv) > 1:
        test_data_file = sys.argv[1]
    else:
        test_data_file = DEFAULT_TEST_DATA_FILE
        print(f"ヒント: 引数にデータファイルのパスを指定すると、別のデータで検証できます。")
        print(f"例: python {os.path.basename(__file__)} <path/to/data.bin>")
    
    if not os.path.exists(test_data_file):
        print(f"[FATAL] Test data file not found: {test_data_file}")
        sys.exit(1)

    print(f"Using data file: {test_data_file}")
    print(f"Using tolerance: rtol={RTOL}, atol={ATOL}\n")

    # ★ 修正点: 表示を揃えるため、一番長いターゲット名の長さを取得
    max_name_len = max(len(name) for name in BENCHMARK_TARGETS.keys())

    results = {}
    for name, command in BENCHMARK_TARGETS.items():
        print(f"Running: {name}...")
        parsed_output = run_and_parse_output(command, test_data_file)
        results[name] = parsed_output

    print("\n--- Verification Results ---")
    if BASE_TARGET_NAME not in results or results[BASE_TARGET_NAME] is None:
        print(f"[FATAL] Base target '{BASE_TARGET_NAME}' failed to run. Cannot compare.")
        return

    base_results = results[BASE_TARGET_NAME]
    all_passed = True

    for name, result_data in results.items():
        if name == BASE_TARGET_NAME:
            continue
        
        if result_data is None:
            # ★ 修正点: f-stringのコロンと不等号で文字の埋め込みと左寄せを指定
            print(f"[ FAIL ] '{name:<{max_name_len}}' failed to run or produce values.")
            all_passed = False
            continue

        max_s_err = calculate_relative_error(base_results["s_values"], result_data["s_values"])
        max_u_err = calculate_relative_error(base_results["u_vector"], result_data["u_vector"])

        s_values_match = np.allclose(base_results["s_values"], result_data["s_values"], rtol=RTOL, atol=ATOL)
        u_vector_match = np.allclose(base_results["u_vector"], result_data["u_vector"], rtol=RTOL, atol=ATOL)

        if s_values_match and u_vector_match:
            # ★ 修正点: 同様に、OKの場合も表示を揃える
            print(f"[  OK  ] '{name:<{max_name_len}}' matches '{BASE_TARGET_NAME}'. (s_err: {max_s_err:.2e}, u_err: {max_u_err:.2e})")
        else:
            all_passed = False
            error_details = []
            if not s_values_match:
                error_details.append(f"s_values mismatch (err: {max_s_err:.2e})")
            if not u_vector_match:
                error_details.append(f"u_vector mismatch (err: {max_u_err:.2e})")
            # ★ 修正点: 同様に、FAILの場合も表示を揃える
            print(f"[ FAIL ] '{name:<{max_name_len}}' does not match '{BASE_TARGET_NAME}'. Reason: {', '.join(error_details)}")

    print("\n------------------------------")
    if all_passed:
        print("✅ All implementations produced consistent results within tolerance.")
    else:
        print("❌ Some implementations produced different results.")

if __name__ == "__main__":
    main()