import numpy as np
import pytest

# Import the dsvdc function from your svd_components module
from python_src.dsvdc import dsvdc

def test_t_svd_fortran_scenario():
    """
    Tests the Python dsvdc function using the specific scenario
    from the Fortran t_svd.f90 program.
    It verifies the numerical correctness of singular values, U, and V.
    Expected U and V are taken from the dsvdc_with_advanced_tester.json Golder Master file.
    Expected S is taken from the t_svd.f90 console output description.
    """
    # 1. Initialize the input matrix 'x' as in t_svd.f90
    # x = [[1, 2, 3],
    #      [4, 5, 6],
    #      [7, 8, 9],
    #      [10, 11, 12]]
    x_input_matrix = np.arange(1.0, 13.0, dtype=np.float64).reshape(4, 3)

    # Define dimensions and job code from t_svd.f90
    n_val = 4  # Number of rows in x
    p_val = 3  # Number of columns in x
    job_val = 11 # Calculate U (full) and V

    # 2. Define expected output values
    # S expected from t_svd.f90 console output (since JSON s is null)
    # Fortran t_svd.f90 comments: "singular values: 25.4624, 1.2907 and zero."
    # Fortran t_svd.f90 FORMAT: (' The calculated singular values = ', 4f10.4/)
    s_expected_from_console = np.array([25.4624, 1.2907, 0.0000])

    # U expected from dsvdc_with_advanced_tester.json (first entry)
    u_expected_from_json = np.array([
        [-0.1408766767502446, -0.8247143517290126, -0.1667836055277655, -0.5217118255580946],
        [-0.3439462942123891, -0.4262639401797497,  0.6112391762414905,  0.5713026075799266],
        [-0.5470159116745336, -0.02781352863048479, -0.7221275358996845,  0.4225302615144269],
        [-0.750085529136678,   0.3706368829187808,  0.2776719651859595, -0.4721210435362597]
    ], dtype=np.float64)

    # V expected from dsvdc_with_advanced_tester.json (first entry)
    v_expected_from_json = np.array([
        [-0.5045331459313139,  0.7607756817814199, -0.4082482904638641],
        [-0.574515704189019,   0.05714051954199287,  0.816496580927726],
        [-0.6444982624467243, -0.6464946426974375, -0.4082482904638622]
    ], dtype=np.float64)
    
    # 3. Pre-allocate arrays for outputs s, e, u, v based on t_svd.f90 declarations
    # Fortran: REAL (dp)  :: e(0:2), x(0:3,0:2), s(0:3), u(0:3,0:3), v(0:2,0:2)
    s_calculated_array = np.empty(n_val, dtype=np.float64)          # To hold s(0:3)
    e_calculated_array = np.empty(p_val, dtype=np.float64)          # To hold e(0:2)
    u_calculated_array = np.empty((n_val, n_val), dtype=np.float64) # To hold u(0:3,0:3)
    v_calculated_array = np.empty((p_val, p_val), dtype=np.float64) # To hold v(0:2,0:2)
    
    # Call your Python dsvdc function.
    # Pass a copy of x_input_matrix because dsvdc in svd_components.py is commented as INTENT(IN OUT) for x.
    try:
        info_calculated = dsvdc(
            x=x_input_matrix.copy(), 
            n=n_val, 
            p=p_val,
            s=s_calculated_array, 
            e=e_calculated_array,
            u=u_calculated_array, 
            v=v_calculated_array,
            job=job_val
        )
    except Exception as e:
        pytest.fail(f"Error calling dsvdc: {e}\n"
                    "Ensure the dsvdc function in svd_components.py is correctly implemented "
                    "and handles inputs/outputs as expected by this test.")

    # 4. Perform assertions
    # Tolerance (atol) is set to 1e-4 as used in the previous failing test display for U.
    # This can be adjusted if necessary.

    # Check info code (typically 0 for success)
    assert info_calculated == 0, f"dsvdc info code indicates an error: {info_calculated}"

    # Compare the first 3 singular values from s_calculated_array
    # (as s_expected_from_console has 3 values corresponding to min(N,P))
    np.testing.assert_allclose(s_calculated_array[:3], s_expected_from_console, atol=1e-4,
                               err_msg="Singular values (s) do not match expected console output values.")
    
    # Compare U matrix against JSON data
    np.testing.assert_allclose(u_calculated_array, u_expected_from_json, atol=1e-4, # Adjusted tolerance
                               err_msg="U matrix does not match expected JSON values.")
    
    # Compare V matrix against JSON data
    np.testing.assert_allclose(v_calculated_array, v_expected_from_json, atol=1e-4, # Adjusted tolerance
                               err_msg="V matrix does not match expected JSON values.")