import sys
import time
import numpy as np
import math

# 注意: setup_pythonpath.ps1 を実行して PYTHONPATH が設定されていることが前提です。
from python_src.dsvdc_vectorized import dsvdc

def load_matrix_from_bin(filepath):
    """
    バイナリファイルから正方行列を読み込む。
    Fortranの列優先順でデータが格納されていると仮定する。
    """
    try:
        with open(filepath, 'rb') as f:
            data = np.fromfile(f, dtype=np.float64)
        
        n = int(math.sqrt(data.size))
        if n * n != data.size:
            raise ValueError("ファイルサイズが正方行列を表していません。")
            
        # Fortran形式のデータを正しく読み込むために order='F' を指定
        matrix = data.reshape((n, n), order='F')
        return matrix
        
    except FileNotFoundError:
        print(f"エラー: ファイルが見つかりません: {filepath}", file=sys.stderr)
        sys.exit(1)
    except ValueError as e:
        print(f"エラー: データ形式が不正です: {e}", file=sys.stderr)
        sys.exit(1)


def main():
    """
    メイン処理
    """
    if len(sys.argv) != 2:
        print(f"使い方: python {sys.argv[0]} <データファイルパス>", file=sys.stderr)
        sys.exit(1)
        
    filepath = sys.argv[1]
    
    x = load_matrix_from_bin(filepath)
    n, p = x.shape
    
    job = 11
    s = np.zeros(min(n, p), dtype=np.float64)
    e = np.zeros(p, dtype=np.float64)
    u = np.zeros((n, n), dtype=np.float64, order='F')
    v = np.zeros((p, p), dtype=np.float64, order='F')
    
    # --- ここからが計測対象 ---
    start_time = time.perf_counter()
    
    info = dsvdc(x, n, p, s, e, u, v, job)
    
    end_time = time.perf_counter()
    # --- ここまでが計測対象 ---
    
    duration = end_time - start_time
    
    if info != 0:
        print(f"警告: dsvdcがエラーコード {info} を返しました。", file=sys.stderr)
    
    # 結果（秒数）のみを標準出力に出力
    print(f"{duration:.8f}")

    # --- Validation Output ---
    print("Top 5 singular values (Python Vectorized):", file=sys.stderr)
    s_list = [f"{val:20.15f}" for val in s[:5]]
    print(" ".join(s_list), file=sys.stderr)

    print("", file=sys.stderr)

    print("First 3 components of leading eigenvector (1st column of U):", file=sys.stderr)
    u_list = [f"{val:20.15f}" for val in u[:3, 0]]
    print(" ".join(u_list), file=sys.stderr)


if __name__ == "__main__":
    main()