import sys
import os
import numpy as np
import time

def get_matrix_from_file(file_path):
    """
    Reads data from a binary file and restores a square matrix.
    """
    if not os.path.exists(file_path):
        print(f"Error: Data file not found at {file_path}", file=sys.stderr)
        sys.exit(1)

    file_size = os.path.getsize(file_path)
    # Each element is a float64 (8 bytes)
    num_elements = file_size // 8
    n_dim = int(np.sqrt(num_elements))

    if n_dim * n_dim != num_elements:
        print(f"Error: Data file size {file_size} does not correspond to a square matrix.", file=sys.stderr)
        sys.exit(1)
        
    matrix = np.fromfile(file_path, dtype=np.float64)
    # Ensure the matrix is in Fortran (column-major) order for f2py
    matrix = matrix.reshape((n_dim, n_dim), order='F')
    return matrix, n_dim

def main():
    """
    Main benchmark process.
    """
    if len(sys.argv) < 2:
        print("Usage: python benchmark_svd_python_f2py.py <path_to_data_file>", file=sys.stderr)
        sys.exit(1)

    data_file = sys.argv[1]
    x, n = get_matrix_from_file(data_file)
    p = n  # For an N x N square matrix, p = n

    try:
        import svd_module_intent_out
        f2py_dsvdc = svd_module_intent_out.svd.dsvdc

    except ImportError as e:
        print("Error: Could not import the dsvdc function from 'svd_module_intent_out'.", file=sys.stderr)
        print(f"Reason: {e}", file=sys.stderr)
        print("Please ensure you have built the module (e.g., 'svd_module_intent_out.pyd') and it's in the correct directory.", file=sys.stderr)
        sys.exit(1)
        
    # --- Prepare for the Fortran subroutine call ---
    # In this f2py version, job=11 is used to calculate s, u, and v.
    job = 11
    
    # --- Time and execute the SVD calculation ---
    start_time = time.perf_counter()
    
    # The 'x' matrix is modified in-place. The output arrays are returned by the function.
    s, e, u, v, info = f2py_dsvdc(x, n, p, job)
    
    end_time = time.perf_counter()

    # --- Output results ---
    elapsed_time = end_time - start_time
    print(f"{elapsed_time:.8f}")
    
    # Optional: Print info for debugging to ensure the call was successful (info=0)
    if info != 0:
        print(f"Warning: dsvdc returned error code {info}", file=sys.stderr)
    
    # --- Validation Output ---
    print("Top 5 singular values (Python f2py):", file=sys.stderr)
    s_list = [f"{val:20.15f}" for val in s[:5]]
    print(" ".join(s_list), file=sys.stderr)

    print("", file=sys.stderr)

    print("First 3 components of leading eigenvector (1st column of U):", file=sys.stderr)
    u_list = [f"{val:20.15f}" for val in u[:3, 0]]
    print(" ".join(u_list), file=sys.stderr)


if __name__ == "__main__":
    main()