import sys
import os
import numpy as np
import time

# --- このプロジェクトのPYTHONPATHが通っていることを前提とする ---
# python_srcディレクトリから自作のctypesラッパーをインポート
try:
    from python_src import svd_ctypes
except ImportError as e:
    print("Error: Could not import the svd_ctypes module.", file=sys.stderr)
    print("Please ensure 'python_src' is in your PYTHONPATH.", file=sys.stderr)
    sys.exit(1)

def get_matrix_from_file(file_path):
    """
    バイナリファイルからデータを読み込み、正方行列を復元する。
    (f2pyのベンチマークから流用)
    """
    if not os.path.exists(file_path):
        print(f"Error: Data file not found at {file_path}", file=sys.stderr)
        sys.exit(1)

    file_size = os.path.getsize(file_path)
    num_elements = file_size // 8
    n_dim = int(np.sqrt(num_elements))

    if n_dim * n_dim != num_elements:
        print(f"Error: Data file size {file_size} does not correspond to a square matrix.", file=sys.stderr)
        sys.exit(1)
        
    matrix = np.fromfile(file_path, dtype=np.float64)
    # Fortranが期待する列優先(F-order)で読み込む
    matrix = matrix.reshape((n_dim, n_dim), order='F')
    return matrix, n_dim

def main():
    """
    メインのベンチマーク処理
    """
    if len(sys.argv) < 2:
        print(f"Usage: python {sys.argv[0]} <path_to_data_file>", file=sys.stderr)
        sys.exit(1)

    data_file = sys.argv[1]
    x, n = get_matrix_from_file(data_file)
    p = n  # N x N の正方行列

    # --- Fortranサブルーチン呼び出しのための配列を準備 ---
    # f2py版のjob=11に合わせる
    job = 11
    
    # 結果を格納する配列をあらかじめ確保する
    s = np.zeros(p, dtype=np.float64)
    e = np.zeros(p, dtype=np.float64)
    # Fortranラッパー dsvdc_c は u(n,n), v(p,p) を期待するのでその通りに確保
    u = np.zeros((n, n), dtype=np.float64, order='F')
    v = np.zeros((p, p), dtype=np.float64, order='F')
    
    # --- 時間を計測してSVD計算を実行 ---
    start_time = time.perf_counter()
    
    # ctypesラッパー関数を呼び出す
    info = svd_ctypes.dsvdc(x, n, p, s, e, u, v, job)
    
    end_time = time.perf_counter()

    # --- 結果を出力 ---
    # run_all_benchmarks.pyがパースできるよう、標準出力には時間だけを出力する
    elapsed_time = end_time - start_time
    print(f"{elapsed_time:.8f}")
    
    # エラーがあれば標準エラー出力に表示
    if info != 0:
        print(f"Warning: SVD calculation failed. info = {info}", file=sys.stderr)

    # --- Validation Output ---
    print("Top 5 singular values (Python ctypes):", file=sys.stderr)
    s_list = [f"{val:20.15f}" for val in s[:5]]
    print(" ".join(s_list), file=sys.stderr)

    print("", file=sys.stderr)

    print("First 3 components of leading eigenvector (1st column of U):", file=sys.stderr)
    u_list = [f"{val:20.15f}" for val in u[:3, 0]]
    print(" ".join(u_list), file=sys.stderr)


if __name__ == "__main__":
    main()
