import sys
import time
import numpy as np
import math

# NOTE: Assumes PYTHONPATH is set by running setup_pythonpath.ps1
from python_src.dsvdc_base_jit import dsvdc


def load_matrix_from_bin(filepath):
    """Load a square matrix from a binary file in Fortran's column-major order."""
    try:
        with open(filepath, 'rb') as f:
            data = np.fromfile(f, dtype=np.float64)
        
        n = int(math.sqrt(data.size))
        if n * n != data.size:
            raise ValueError("File size does not represent a square matrix.")
            
        # 'F' order is crucial to correctly interpret Fortran-ordered data.
        matrix = data.reshape((n, n), order='F')
        return matrix
        
    except FileNotFoundError:
        print(f"Error: File not found: {filepath}", file=sys.stderr)
        sys.exit(1)
    except ValueError as e:
        print(f"Error: Invalid data format: {e}", file=sys.stderr)
        sys.exit(1)


def main():
    if len(sys.argv) != 2:
        print("Usage: python benchmark_svd_python_base.py <data_file_path>", file=sys.stderr)
        sys.exit(1)
        
    filepath = sys.argv[1]
    
    x = load_matrix_from_bin(filepath)
    n, p = x.shape
    
    job = 11
    s = np.zeros(min(n, p), dtype=np.float64)
    e = np.zeros(p, dtype=np.float64)
    u = np.zeros((n, n), dtype=np.float64, order='F')
    v = np.zeros((p, p), dtype=np.float64, order='F')
    
    start_time = time.perf_counter()
    info = dsvdc(x, n, p, s, e, u, v, job)
    end_time = time.perf_counter()
    
    duration = end_time - start_time
    
    if info != 0:
        print(f"Warning: dsvdc returned error code {info}", file=sys.stderr)
    
    print(f"{duration:.8f}")

    # --- Validation Output ---
    print("Top 5 singular values (Python Base_JIT):", file=sys.stderr)
    # Use "{val:20.15f}" to match Fortran's F20.15 fixed-width format.
    s_list = [f"{val:20.15f}" for val in s[:5]]
    print(" ".join(s_list), file=sys.stderr)

    print("", file=sys.stderr)

    print("First 3 components of leading eigenvector (1st column of U):", file=sys.stderr)
    u_list = [f"{val:20.15f}" for val in u[:3, 0]]
    print(" ".join(u_list), file=sys.stderr)


if __name__ == "__main__":
    main()
