# Test_SVD_Advanced_using_DSVDC.py
# Fortranのテストプログラム Test_DSVDC_Advanced.f90 のPython版

import numpy as np

try:
    # Import the compiled f2py module, typically named svd_module
    import svd_module

    # f2py can place the function in different locations depending on the module structure.
    # We check for both common possibilities to make the script robust.
    if hasattr(svd_module, 'svd') and hasattr(svd_module.svd, 'dsvdc'):
        dsvdc = svd_module.svd.dsvdc
    elif hasattr(svd_module, 'dsvdc'):
        dsvdc = svd_module.dsvdc
    else:
        raise AttributeError("dsvdc function not found in the svd_module.")

except (ImportError, AttributeError) as e:
    print("Error: Could not import or find the dsvdc function from the f2py 'svd_module'.")
    print(f"Reason: {e}")
    print("Please ensure you have built the f2py module and it is in your Python path.")
    exit()

def print_matrix_info(mat, job, title):
    """入力行列の情報を表示するヘルパー関数"""
    try:
        n, p = mat.shape
    except ValueError:
        n, p = (mat.shape[0], 1) if mat.ndim == 1 else (1, 1)

    print(title)
    print(f"N={n}, P={p}, JOB={job}")
    print("Matrix elements (first 5x5 if large):")

    rows_to_print = min(n, 5)
    cols_to_print = min(p, 5)
    
    row_format = " ".join(["{:12.4E}" for _ in range(cols_to_print)])
    
    mat_2d = np.atleast_2d(mat)
    for i in range(rows_to_print):
        print(row_format.format(*mat_2d[i, :cols_to_print]))
    if n > 5 or p > 5:
        print("...")
    print()

def print_svd_results(s, e, u, v, info, job, n, p):
    """dsvdcの実行結果を表示するヘルパー関数"""
    print("SVD Results:")
    print(f"INFO = {info}")
    if info != 0:
        print(f"*** WARNING: DSVDC did not converge completely. INFO = {info}")

    num_sv_to_print = min(n, p)
    s_format = " ".join(["{:12.4E}" for _ in range(num_sv_to_print)])
    print(f"Singular values S = {s_format.format(*s[:num_sv_to_print])}")
    
    e_format = " ".join(["{:12.4E}" for _ in range(p)])
    print(f"Superdiagonal E (full) = {e_format.format(*e[:p])}")
    print()

    job_str = f"{job:02d}"
    jobu = int(job_str[0])
    jobv = int(job_str[1])
    
    if jobu == 0:
        print("Matrix U was not computed (JOB).")
    else:
        print("Matrix U (first 5x5 if large):")
        ncu_actual = n if jobu == 1 else min(n,p)
        rows_to_print = min(n, 5)
        cols_to_print = min(ncu_actual, 5)
        row_format = " ".join(["{:8.3f}" for _ in range(cols_to_print)])
        if jobu == 1:
            print(f"(Computed as N x N = {n}x{n})")
        elif jobu == 2:
            print(f"(Computed as N x min(N,P) = {n}x{ncu_actual})")
        
        u_view = u[:, :ncu_actual]
        for i in range(rows_to_print):
            print(row_format.format(*u_view[i, :cols_to_print]))
        if n > 5 or ncu_actual > 5:
            print("...")
    print()
    
    if jobv == 0:
        print("Matrix V was not computed (JOB).")
    else:
        print("Matrix V (first 5x5 if large):")
        print(f"(Computed as P x P = {p}x{p})")
        rows_to_print = min(p, 5)
        cols_to_print = min(p, 5)
        row_format = " ".join(["{:8.3f}" for _ in range(cols_to_print)])
        
        for i in range(rows_to_print):
            print(row_format.format(*v[i, :cols_to_print]))
        if p > 5:
            print("...")
    print()

def run_test(test_count, n, p, job, x_init_func, title, expected_msg=""):
    """テストケースを実行するための共通関数"""
    print(f"--- Test Case {test_count}: {title} (N={n}, P={p}, JOB={job}) ---")
    
    x = np.zeros((n, p), dtype=float, order='F')
    x_init_func(x, n, p)
    
    s = np.zeros(max(n, p), dtype=float)
    e = np.zeros(p, dtype=float)
    u = np.zeros((n, n), dtype=float, order='F')
    v = np.zeros((p, p), dtype=float, order='F')
    
    print_matrix_info(x, job, "Input Matrix X:")
    info = dsvdc(x, n, p, s, e, u, v, job)
    print_svd_results(s, e, u, v, info, job, n, p)
    
    if expected_msg:
        print(expected_msg)
    print('--------------------------------------------------------------\n')
    return test_count + 1

def main():
    """全てのテストケースを実行するメイン関数"""
    print('==============================================================')
    print('DSVDC Advanced Test Program START for f2py')
    print('==============================================================\n')
    
    test_count = 1

    # Test Case 1: Original Example
    # CORRECTED: Fills row-by-row, same as original Fortran test.
    def init1(x, n, p):
        x[:] = np.arange(1.0, n * p + 1.0).reshape((n, p))
    test_count = run_test(test_count, 4, 3, 11, init1, "Original Example")

    # Test Case 2: Square Well-Conditioned
    def init2(x, n, p):
        x[:,:] = [[4.0, 1.0, -1.0], [1.0, 2.0, 1.0], [-1.0, 1.0, 3.0]]
    test_count = run_test(test_count, 3, 3, 11, init2, "Square Well-Conditioned")

    # Test Case 3: Tall Matrix
    # CORRECTED: Correctly initializes the matrix based on Fortran logic.
    def init3(x, n, p):
        for j in range(p):
            x[:, j] = np.arange(1, n + 1) + j * n
    test_count = run_test(test_count, 5, 2, 11, init3, "Tall Matrix")

    # Test Case 4: Wide Matrix
    def init4(x, n, p):
        x[:] = np.arange(1.0, n * p + 1.0).reshape((n, p))
    test_count = run_test(test_count, 2, 5, 11, init4, "Wide Matrix")

    # Test Case 5: Rank Deficient
    def init5(x, n, p):
        x[:, 0] = [1.0, 2.0, 3.0]
        x[:, 1] = [4.0, 5.0, 6.0]
        x[:, 2] = x[:, 0] + x[:, 1]
    test_count = run_test(test_count, 3, 3, 11, init5, "Rank Deficient", 
                          "Expected: One singular value should be close to zero.")

    # Test Case 6: Zero Matrix
    def init6(x, n, p):
        x.fill(0.0)
    test_count = run_test(test_count, 3, 2, 11, init6, "Zero Matrix",
                          "Expected: All singular values should be close to zero.")
                          
    # Test Case 7: Diagonal Matrix
    def init7(x, n, p):
        np.fill_diagonal(x, [5.0, -2.0, 1.0])
    test_count = run_test(test_count, 3, 3, 11, init7, "Diagonal Matrix",
                          "Expected: Singular values should be ABS of diagonal entries (5, 2, 1), possibly reordered.")

    # Test Case 8: Diagonal Matrix with repeated SVs
    def init8(x, n, p):
        np.fill_diagonal(x, [2.0, -2.0, 1.0])
    test_count = run_test(test_count, 3, 3, 11, init8, "Diagonal Matrix with repeated SVs",
                          "Expected: Singular values should be (2, 2, 1), possibly reordered.")

    # Test Case 9: 1x1 Matrix
    def init9(x, n, p):
        x[0,0] = -7.0
    test_count = run_test(test_count, 1, 1, 11, init9, "1x1 Matrix", "Expected: Singular value should be 7.")

    # Test Case 10: 2x1 Matrix
    def init10(x, n, p):
        x[:,0] = [3.0, 4.0]
    test_count = run_test(test_count, 2, 1, 11, init10, "2x1 Matrix (Column Vector)", "Expected: Singular value should be 5.")

    # Test Case 11: 1x2 Matrix
    def init11(x, n, p):
        x[0,:] = [3.0, 4.0]
    test_count = run_test(test_count, 1, 2, 11, init11, "1x2 Matrix (Row Vector)", "Expected: Singular value should be 5.")

    # Test Case 12: Ill-conditioned like
    def init12(x, n, p):
        np.fill_diagonal(x, [1.0E8, 1.0E-8])
    test_count = run_test(test_count, 2, 2, 11, init12, "Ill-conditioned like",
                          "Expected: Singular values 1E8 and 1E-8. Check convergence (info=0).")
    
    # --- Testing different JOB values ---
    print('--- Testing different JOB values with N=4, P=3 matrix ---\n')
    job_values = {'00 (S only)': 0, 
                  '01 (S, V)': 1, 
                  '10 (S, U full)': 10, 
                  '21 (S, U thin, V)': 21}

    for desc, job_val in job_values.items():
        # The initialization function is the same as the corrected Test Case 1
        test_count = run_test(test_count, 4, 3, job_val, init1, f"JOB Test: {desc}")

    print('==============================================================')
    print('DSVDC Advanced Test Program END')
    print('==============================================================')


if __name__ == "__main__":
    main()