Module recommendation_module
    Use, Intrinsic :: iso_fortran_env, Only: int64, real64
    Implicit None
    Private
    Public :: recommendation_result
    Public :: recommend_from_book_id

    Type :: recommendation_result
        Integer (int64) :: book_id
        Character (Len=100) :: title
        Real (real64) :: predicted_rating
    End Type recommendation_result

Contains
    Subroutine recommend_from_book_id(book_id, u, sigma, vt, books_file, book_id_map, n_recommendations, book_title, &
        recommendations)
        Integer (int64), Intent (In) :: book_id
        Real (real64), Intent (In) :: u(:, :), sigma(:), vt(:, :)
        Character (Len=*), Intent (In) :: books_file
        Integer (int64), Intent (In) :: book_id_map(:)
        Integer, Intent (In) :: n_recommendations
        Character (Len=*), Intent (Out) :: book_title
        Type (recommendation_result), Allocatable, Intent (Out) :: recommendations(:)
        Integer :: book_index, i
        Real (real64), Allocatable :: new_user_features(:), estimated_user_features(:)
        Real (real64), Allocatable :: predictions(:), temp_features(:)
        Real (real64), Allocatable :: u_transpose(:, :), sigma_inv(:)
        Real (real64) :: rcond, cutoff
        Logical :: found = .False.
        Integer :: num_books, num_features

        num_books = size(vt, 2)
        num_features = size(vt, 1)
        Do i = 1, size(book_id_map)
            If (book_id_map(i)==book_id) Then
                book_index = i
                found = .True.
                Exit
            End If
        End Do
        If (.Not. found) Then
            Write (*, '(A,I0,A)') 'BookID ', book_id, ' ܂B'
            Return
        End If
        Call get_book_title(books_file, book_id, book_title)
        Allocate (new_user_features(num_books))
        new_user_features = 0.0_real64
        new_user_features(book_index) = 5.0_real64
        Allocate (temp_features(num_features))
        temp_features = matmul(new_user_features, transpose(vt))
        Allocate (sigma_inv(num_features))
        sigma_inv = 0.0_real64
        rcond = 1.0E-15_real64
        cutoff = rcond*maxval(sigma)
        Where (sigma>cutoff)
            sigma_inv = 1.0_real64/sigma
        Elsewhere
            sigma_inv = 0.0_real64
        End Where
        temp_features = temp_features*sigma_inv
        Allocate (u_transpose(size(u,2),size(u,1)))
        Call transpose_matrix(u, u_transpose)
        Allocate (estimated_user_features(size(u,1)))
        estimated_user_features = matmul(temp_features, u_transpose)
        temp_features = matmul(estimated_user_features, u)
        temp_features = temp_features*sigma
        Allocate (predictions(num_books))
        predictions = matmul(temp_features, vt)
        predictions(book_index) = -huge(0.0_real64)
        Allocate (recommendations(n_recommendations))
        Call get_top_recommendations(predictions, n_recommendations, book_id_map, books_file, recommendations)
        Deallocate (new_user_features, estimated_user_features, predictions)
        Deallocate (temp_features, u_transpose, sigma_inv)
    End Subroutine recommend_from_book_id

    Subroutine get_top_recommendations(predictions, n, book_id_map, books_file, results)
        Real (real64), Intent (In) :: predictions(:)
        Integer, Intent (In) :: n
        Integer (int64), Intent (In) :: book_id_map(:)
        Character (Len=*), Intent (In) :: books_file
        Type (recommendation_result), Intent (Out) :: results(:)

        Integer, Allocatable :: indices(:)
        Integer :: i, j, max_idx
        Real (real64) :: max_val

        ! CfbNX̏
        Allocate (indices(size(predictions)))
        Do i = 1, size(predictions)
            indices(i) = i
        End Do

        ! nIiI\[gj
        Do i = 1, n
            max_val = predictions(indices(i))
            max_idx = i
            Do j = i + 1, size(predictions)
                If (predictions(indices(j))>max_val) Then
                    max_val = predictions(indices(j))
                    max_idx = j
                End If
            End Do
            If (max_idx/=i) Then
                j = indices(i)
                indices(i) = indices(max_idx)
                indices(max_idx) = j
            End If
        End Do

        ! ʂ̐
        Do i = 1, n
            results(i)%book_id = book_id_map(indices(i))
            results(i)%predicted_rating = predictions(indices(i))
            Call get_book_title(books_file, results(i)%book_id, results(i)%title)
        End Do

        Deallocate (indices)
    End Subroutine get_top_recommendations

    Subroutine get_book_title(books_file, book_id, title)
        Character (Len=*), Intent (In) :: books_file
        Integer (int64), Intent (In) :: book_id
        Character (Len=*), Intent (Out) :: title

        Integer :: io_status, tab_pos
        Character (Len=1000) :: line
        Integer (int64) :: temp_id
        Logical :: found_book = .False.

        Open (Unit=21, File=books_file, Status='old', Action='read', Iostat=io_status)
        If (io_status/=0) Then
            Write (*, '(A)') 'Error: Cannot open books file'
            title = 'Unknown Title'
            Return
        End If

        Read (21, '(A)') line      ! wb_[XLbv
        Do
            Read (21, '(A)', Iostat=io_status) line
            If (io_status/=0) Exit

            tab_pos = index(line, char(9))
            If (tab_pos>0) Then
                Read (line(1:tab_pos-1), *) temp_id
                If (temp_id==book_id) Then
                    title = adjustl(line(tab_pos+1:))
                    found_book = .True.
                    Exit
                End If
            End If
        End Do

        If (.Not. found_book) Then
            title = 'Unknown Title'
        End If

        Close (21)
    End Subroutine get_book_title

    Subroutine transpose_matrix(input_matrix, output_matrix)
        Real (real64), Intent (In) :: input_matrix(:, :)
        Real (real64), Intent (Out) :: output_matrix(:, :)
        Integer :: i, j

        Do i = 1, size(input_matrix, 1)
            Do j = 1, size(input_matrix, 2)
                output_matrix(j, i) = input_matrix(i, j)
            End Do
        End Do
    End Subroutine transpose_matrix

End Module recommendation_module
